#include <stdio.h>
#include <string.h>
#include <math.h>
#include "focas_util.h"
#include "fio.h"

int read_distpars(char *fname, struct distortion *distpar)
{
	FILE	*fp;
	char	buf[1024];
	int	i=0,nread=0;
	double	pars[NDISTPARS];

	if((fp=fopen(fname,"rt"))==NULL)
	{
		fprintf(stderr,"cannot open %s!\n",fname);
		perror("fopen");
		return DC_ERR;
	}

	while(fgets(buf,sizeof(buf),fp)!=NULL && i<NDISTPARS)
	{
		nread+=sscanf(buf,"%lf",&pars[i]);
		i++;
	}
	fclose(fp);

	if(nread!=NDISTPARS)
	{
		fprintf(stderr,"parameter file read error!\n");
		return DC_ERR;
	}

	distpar->yshift=pars[0];
	distpar->gapsize=pars[1];
	distpar->diffrot=pars[2];
	distpar->beta=pars[3];
	distpar->dist_cenx=pars[4];
	distpar->dist_ceny=pars[5];
	distpar->mask_cenx=pars[6];
	distpar->mask_ceny=pars[7];
	distpar->mask_rot_cenx=pars[8];
	distpar->mask_rot_ceny=pars[9];
	distpar->rotator_cenx=pars[10];
	distpar->rotator_ceny=pars[11];
	distpar->coef[0]=pars[12];
	distpar->coef[1]=pars[13];
	distpar->coef[2]=pars[14];
	distpar->coef[3]=pars[15];
	distpar->theta_offset=pars[16];
// caution!!!  "+1.0"
	distpar->coefinv[0]=pars[17]+1.0;
//
	distpar->coefinv[1]=pars[18];
	distpar->coefinv[2]=pars[19];
	distpar->coefinv[3]=pars[20];
	distpar->overscan=(int)pars[21];

	return DC_OK;
}

// sbr in mm
// mdp in pixel (distortion is not applied)
int sbr2mdp(double *x, double *y, struct distortion distpar)
{
	double	r,theta;

	r=sqrt(*x * *x + *y * *y);
	theta=atan2(*y,*x);

	// offset rotation
	theta-=distpar.theta_offset/RAD;

	// convert to CCD pixel scale
	r*=distpar.beta/0.015;

	// to CCD coordinates
	*x=r*cos(theta)+distpar.mask_cenx;
	*y=r*sin(theta)+distpar.mask_ceny;

	return FCS_OK;
}

// with correction for MOS mask rotation
int sbr2mdp_rot(double *x, double *y, struct distortion distpar, double mask_rot)
{
	double	r,theta;

	r=sqrt(*x * *x + *y * *y);
	theta=atan2(*y,*x);

	// offset rotation
	theta-=distpar.theta_offset/RAD;

	// mask rotation from Falcon
	theta-=mask_rot/RAD;

	// convert to CCD pixel scale
	r*=distpar.beta/0.015;

	// to CCD coordinates
	*x=r*cos(theta)+distpar.mask_cenx;
	*y=r*sin(theta)+distpar.mask_ceny;

	return FCS_OK;
}

int readsbr(char *fname, struct sbr_parameters *slit, int *nslit)
{
	FILE	*fp;
	char	buf[BUFSIZE];
	double	x1,x2,y1,y2,width;
	int	i;

	if((fp=fopen(fname,"rt"))==NULL)
	{
		fprintf(stderr,"cannot open %s!\n",fname);
		return FCS_ERR;
	}

	i=0;
	while(fgets(buf,sizeof(buf),fp)!=NULL)
	{
		if(strncmp(buf,"B,",2)==0)
		{
			if(sscanf(buf,"B, %lf, %lf, %lf, %lf, %lf",
					&x1,&y1,&x2,&y2,&width)!=5)
			{
				fprintf(stderr,"#warning! unknown format : %s",buf);
				continue;
			}
			slit[i].type='B';
			slit[i].x1=x1;
			slit[i].x2=x2;
			slit[i].y1=y1;
			slit[i].y2=y2;
			slit[i].width=width;
			i++;
		}
		else if(strncmp(buf,"C,",2)==0)
		{
			if(sscanf(buf,"C, %lf, %lf, %lf", &x1,&y1,&width)!=3)
			{
				fprintf(stderr,"#warning! unknown format : %s",buf);
				continue;
			}
			slit[i].type='C';
			slit[i].x1=slit[i].x2=x1;
			slit[i].y1=slit[i].y2=y1;
			slit[i].width=width;
			i++;
		}
		else
			continue;

		if(i>=*nslit)
		{
			fprintf(stderr,"#warning! number of slits is >%d!\n",*nslit);
			break;
		}
	}
	fclose(fp);

	*nslit=i;

	return FCS_OK;
}

int readmdp(char *fname, struct mdp_parameters slit[], int *nslit)
{
	FILE	*fp;
	char	buf[BUFSIZE],type[BUFSIZE],*cp;
	double	x,y,length,width,angle,priority;
	int	i,j;

	if((fp=fopen(fname,"rt"))==NULL)
	{
		fprintf(stderr,"cannot open %s!\n",fname);
		return FCS_ERR;
	}

	i=0;
	while(fgets(buf,sizeof(buf),fp)!=NULL)
	{
		if(sscanf(buf,"%lf %lf %lf %lf %lf %lf %s",&x,&y,
				&length,&width,&angle,&priority,type)!=7)
			continue;

		slit[i].x=x;
		slit[i].y=y;
		slit[i].length=length;
		slit[i].width=width;
		slit[i].angle=angle;
		slit[i].priority=priority;

		if(type[0]=='B')
		{
			slit[i].type='B';

			if((cp=strchr(buf,'B'))!=NULL)
			{
				cp+=2;
				j=0;
				while(*cp!='\n'&&*cp!='\0')
				{
					slit[i].comment[j]=*cp;
					cp++;
					j++;
				}
				slit[i].comment[j]='\0';
			}
		}
		else if(type[0]=='C')
		{
			slit[i].type='C';

			if((cp=strchr(buf,'C'))!=NULL)
			{
				cp+=2;
				j=0;
				while(*cp!='\n'&&*cp!='\0')
				{
					slit[i].comment[j]=*cp;
					cp++;
					j++;
				}
				slit[i].comment[j]='\0';
			}
		}
		else
		{
			fprintf(stderr,"#warning! unknown format:%s",buf);
			continue;
		}

		i++;

		if(i>=*nslit)
		{
			fprintf(stderr,"#warning! the number of slits is >%d!\n",*nslit);
			break;
		}
	}
	fclose(fp);

	*nslit=i;

	return FCS_OK;
}

int read_headpars(char *fname, struct header *headpar)
{
	long	xlen,ylen;
	int	xbin,ybin,chip;
	char	buf[64];

	if(getxylen(fname,&xlen,&ylen)!=FCS_OK)
		return FCS_ERR;

	if(get_fits_head(fname,"BIN-FCT1",buf)!=FCS_OK)
		return FCS_ERR;
	if(sscanf(buf,"%d",&xbin)!=1)
		return FCS_ERR;

	if(get_fits_head(fname,"BIN-FCT2",buf)!=FCS_OK)
		return FCS_ERR;
	if(sscanf(buf,"%d",&ybin)!=1)
		return FCS_ERR;

	if(get_fits_head(fname,"DET-ID",buf)!=FCS_OK)
		return FCS_ERR;
	if(sscanf(buf,"%d",&chip)!=1)
		return FCS_ERR;

	if(get_fits_head(fname,"DISPERSR",headpar->grism)!=FCS_OK)
		strcpy(headpar->grism,"None");

	if(get_fits_head(fname,"FILTER01",buf)!=FCS_OK)
		return FCS_ERR;
	if(strcasecmp(buf,"none")==0)
	{
		if(get_fits_head(fname,"FILTER02",buf)!=FCS_OK)
			return FCS_ERR;
		if(strcasecmp(buf,"none")==0)
		{
			if(get_fits_head(fname,"FILTER03",buf)!=FCS_OK)
				return FCS_ERR;
		}
	}
	strcpy(headpar->filter,buf);

	headpar->xlen=xlen;
	headpar->ylen=ylen;
	headpar->xbin=xbin;
	headpar->ybin=ybin;
	headpar->chip=chip;

	return FCS_OK;
}

int get_specmode(char *data_directory, char *grism_code, char *filter_code, struct spmode *mode)
{
	FILE	*fp;
	char	buf[BUFSIZE];
	char	fname[BUFSIZE];
	char	filter[16]="", grism[16]="";
	double	wc,w1,w2,dispersion,dx;
	const char	func[]="get_specmode()";

	sprintf(fname,"%s/%s",data_directory,BARCODE);
	if((fp=fopen(fname,"rt"))==NULL)
	{
		fprintf(stderr,"%s : cannot open %s!\n",func,fname);
		return FCS_ERR;
	}

	while(fgets(buf,sizeof(buf),fp)!=NULL)
	{
		if(buf[0]=='#')
			continue;

		if(strstr(buf,grism_code)!=NULL)
			sscanf(buf,"%s ",grism);
		if(strstr(buf,filter_code)!=NULL)
			sscanf(buf,"%s ",filter);
	}
	fclose(fp);

	if(strcmp(grism,"")==0)
	{
		fprintf(stderr,"%s : grism (%s) was not found.\n",func,grism_code);
		return FCS_ERR;
	}
	if(strcmp(filter,"")==0)
	{
		fprintf(stderr,"%s : filter (%s) was not found.\n",func,filter_code);
		return FCS_ERR;
	}

	strcpy(mode->grism,grism);
	strcpy(mode->filter,filter);

	sprintf(fname,"%s/%s",data_directory,SPECMODE);
	if((fp=fopen(fname,"rt"))==NULL)
	{
		fprintf(stderr,"%s : cannot open %s!\n",func,fname);
		return FCS_ERR;
	}

	while(fgets(buf,sizeof(buf),fp)!=NULL)
	{
		if(buf[0]=='#')
			continue;

		if(sscanf(buf,"%s %s %lf %lf %lf %lf %lf",grism,filter,&wc,&w1,&w2,
				&dispersion,&dx)!=7)
		{
			fprintf(stderr,"%s : warning, format error in %s (%s)",func,
					SPECMODE,buf);
			continue;
		}

		if(strcasecmp(mode->grism,grism)==0 && strcasecmp(mode->filter,filter)==0)
		{
			mode->wc=wc;
			mode->w1=w1;
			mode->w2=w2;
			mode->dispersion=dispersion;
			mode->dx=dx;

			fclose(fp);
			return FCS_OK;
		}
	}
	fclose(fp);

	fprintf(stderr,"%s : spec. mode with %s and %s was not found.\n",func,mode->grism,mode->filter);
	return FCS_ERR;
}
