#include <stdio.h>
#include <fitsio.h>
#include <malloc.h>
#include <math.h>
#include "distcalib.h"

/*
 parameters:
	idata : input data array
	odata : output array to write
	headpar : header information (axlen, binning, chip No.)
	distpar : distortion parameters
	delta_x : in binned pixel, >0 if resulting image moves to right
	delta_y : in binned pixel, >0 if resulting image moves up

#define COL_CAM 1.25
 COL_CAM : parameter for spectroscopic data
	   used in the transformation from CCD pixel to tel-focus plane

 2006/11/11
 remove COL_CAM and add col_cam to the parameter of distcalib()

 2010/7/23
 add delta_x and delta_y for the parameters
*/


int distcalib(float *idata, float *odata, struct header headpar,
		struct distortion distpar, double delta_x, double delta_y, double col_cam)
{
	double	xcen,ycen;
	long	lx,ly;
	long	nx,ny;
	double	xpos[4],ypos[4];
	int	i;
	double	value,area;
	double	xfraction,yfraction;

	if(headpar.chip==1)
	{
		xcen=distpar.dist_cenx-(2047-distpar.overscan)-distpar.gapsize+distpar.overscan;
		ycen=distpar.dist_ceny;
	}
	else
	{
		xcen=distpar.dist_cenx;
		ycen=distpar.dist_ceny-distpar.yshift;
	}
printf("#chip=%d, center=(%lf,%lf)\n",headpar.chip,xcen,ycen);

	for(ly=1;ly<=headpar.ylen;ly++)
	{
		if(ly%500==0) printf("%ld/%ld\n",ly,headpar.ylen);

		for(lx=1;lx<=headpar.xlen;lx++)
		{
			xpos[0]=lx-0.5; //lower-left, point0
			ypos[0]=ly-0.5; //lower-left, point0
			xpos[1]=lx+0.5; //lower-right, point1
			ypos[1]=ly-0.5; //lower-right, point1
			xpos[2]=lx+0.5; //upper-right, point2
			ypos[2]=ly+0.5; //upper-right, point2
			xpos[3]=lx-0.5; //upper-left, point3
			ypos[3]=ly+0.5; //upper-left, point3

#ifdef DEBUG
	printf("#(%ld,%ld)\n",lx,ly);
#endif
			for(i=0;i<4;i++)
			{
				// rotation (overwrite)
				if(distpar.diffrot!=0.0)
				{
					rotate(&xpos[i],&ypos[i],xcen/headpar.xbin,ycen/headpar.ybin,
							distpar.diffrot/headpar.xbin*headpar.ybin);
				}
				// image to tel-focus scale
				xpos[i]=xpos[i]*headpar.xbin-xcen;
				xpos[i]*=0.015/distpar.beta;
				ypos[i]=ypos[i]*headpar.ybin-ycen;
				ypos[i]*=0.015/distpar.beta/col_cam;
				// distortion correction (overwrite)
				distcor(&xpos[i],&ypos[i],distpar);

				// tel-focus to distortion corrected image coords.
				xpos[i]/=0.015/distpar.beta;
//				xpos[i]=(xpos[i]+xcen)/headpar.xbin;
				xpos[i]=(xpos[i]+xcen)/headpar.xbin-delta_x;
				ypos[i]/=0.015/distpar.beta/col_cam;
//				ypos[i]=(ypos[i]+ycen)/headpar.ybin;
				ypos[i]=(ypos[i]+ycen)/headpar.ybin-delta_y;
#ifdef DEBUG
	printf("#%d:(%lf,%lf)\n",i,xpos[i],ypos[i]);
#endif
			}

			// calc pixel value (assume almost rectangular shape)
			value=0.0;
			area=0.0;
			//lower left
			nx=(long)floor(xpos[0]+0.5);
			ny=(long)floor(ypos[0]+0.5);
			xfraction=nx+0.5-xpos[0];
			yfraction=ny+0.5-ypos[0];
			if(nx>=1 && nx<=headpar.xlen && ny>=1 && ny<=headpar.ylen)
			{
				value+=idata[nx-1+headpar.xlen*(ny-1)]*xfraction*yfraction;
				area+=xfraction*yfraction;
#ifdef DEBUG
	printf("%lf %lf\n",idata[nx-1+headpar.xlen*(ny-1)]*xfraction*yfraction,xfraction*yfraction);
#endif
			}
			//low (use mean y of point0 and point1)
			ny=(long)floor(ypos[0]+0.5);
			yfraction=ny+0.5-(ypos[0]+ypos[1])/2.0;
			for(nx=(long)floor(xpos[0]+0.5)+1;nx<=ceil(xpos[1]-0.5)-1;nx++)
			{
				if(nx>=1 && nx<=headpar.xlen && ny>=1 && ny<=headpar.ylen)
				{
					value+=idata[nx-1+headpar.xlen*(ny-1)]*yfraction;
					area+=yfraction;
#ifdef DEBUG
	printf("%lf %lf\n",idata[nx-1+headpar.xlen*(ny-1)]*yfraction,yfraction);
#endif
				}
			}
			//lower right
			nx=(long)ceil(xpos[1]-0.5);
			ny=(long)floor(ypos[1]+0.5);
			xfraction=xpos[1]-nx+0.5;
			yfraction=ny+0.5-ypos[1];
			if(nx>=1 && nx<=headpar.xlen && ny>=1 && ny<=headpar.ylen)
			{
				value+=idata[nx-1+headpar.xlen*(ny-1)]*xfraction*yfraction;
				area+=xfraction*yfraction;
#ifdef DEBUG
	printf("%lf %lf\n",idata[nx-1+headpar.xlen*(ny-1)]*xfraction*yfraction,xfraction*yfraction);
#endif
			}
			//right (use mean x of point1 and point2)
			nx=(long)ceil(xpos[1]-0.5);
			xfraction=(xpos[1]+xpos[2])/2.0-nx+0.5;
			for(ny=(long)floor(ypos[1]+0.5)+1;ny<=ceil(ypos[2]-0.5)-1;ny++)
			{
				if(nx>=1 && nx<=headpar.xlen && ny>=1 && ny<=headpar.ylen)
				{
					value+=idata[nx-1+headpar.xlen*(ny-1)]*xfraction;
					area+=xfraction;
#ifdef DEBUG
	printf("%lf %lf\n",idata[nx-1+headpar.xlen*(ny-1)]*xfraction,xfraction);
#endif
				}
			}
			//upper right
			nx=(long)ceil(xpos[2]-0.5);
			ny=(long)ceil(ypos[2]-0.5);
			xfraction=xpos[2]-nx+0.5;
			yfraction=ypos[2]-ny+0.5;
			if(nx>=1 && nx<=headpar.xlen && ny>=1 && ny<=headpar.ylen)
			{
				value+=idata[nx-1+headpar.xlen*(ny-1)]*xfraction*yfraction;
				area+=xfraction*yfraction;
#ifdef DEBUG
	printf("%lf %lf\n",idata[nx-1+headpar.xlen*(ny-1)]*xfraction*yfraction,xfraction*yfraction);
#endif
			}
			//up (use mean y of point2 and point3)
			ny=(long)ceil(ypos[2]-0.5);
			yfraction=(ypos[2]+ypos[3])/2.0-ny+0.5;
			for(nx=(long)floor(xpos[3]+0.5)+1;nx<=ceil(xpos[2]-0.5)-1;nx++)
			{
				if(nx>=1 && nx<=headpar.xlen && ny>=1 && ny<=headpar.ylen)
				{
					value+=idata[nx-1+headpar.xlen*(ny-1)]*yfraction;
					area+=yfraction;
#ifdef DEBUG
	printf("%lf %lf\n",idata[nx-1+headpar.xlen*(ny-1)]*yfraction,yfraction);
#endif
				}
			}
			//upper left
			nx=(long)floor(xpos[3]+0.5);
			ny=(long)ceil(ypos[3]-0.5);
			xfraction=nx+0.5-xpos[3];
			yfraction=ypos[3]-ny+0.5;
			if(nx>=1 && nx<=headpar.xlen && ny>=1 && ny<=headpar.ylen)
			{
				value+=idata[nx-1+headpar.xlen*(ny-1)]*xfraction*yfraction;
				area+=xfraction*yfraction;
#ifdef DEBUG
	printf("%lf %lf\n",idata[nx-1+headpar.xlen*(ny-1)]*xfraction*yfraction,xfraction*yfraction);
#endif
			}
			//left (use mean x of point0 and point3)
			nx=(long)floor(xpos[3]+0.5);
			xfraction=nx+0.5-(xpos[0]+xpos[3])/2.0;
			for(ny=(long)floor(ypos[0]+0.5)+1;ny<=ceil(ypos[3]-0.5)-1;ny++)
			{
				if(nx>=1 && nx<=headpar.xlen && ny>=1 && ny<=headpar.ylen)
				{
					value+=idata[nx-1+headpar.xlen*(ny-1)]*xfraction;
					area+=xfraction;
#ifdef DEBUG
	printf("%lf %lf\n",idata[nx-1+headpar.xlen*(ny-1)]*xfraction,xfraction);
#endif
				}
			}

			if(area!=0.0)
				odata[lx-1+headpar.xlen*(ly-1)]=(float)(value/area);
			else
				odata[lx-1+headpar.xlen*(ly-1)]=0.0;
		}
	}

	return DC_OK;
}

// dx,dy : in mm
void distcor(double *dx, double *dy, struct distortion distpar)
{
	int	i;
	double	radius,magnify;

	radius=sqrt(*dx * *dx + *dy * *dy);
	magnify=0.0;
	for(i=0;i<4;i++)
		magnify+=distpar.coefinv[i]*pow(radius,(double)i);

	*dx *= magnify;
	*dy *= magnify;
}

// angle>0.0 if resulting image rotates counter-clockwise
void rotate(double *x, double *y, double xcen, double ycen, double angle)
{
	double	dx,dy;

	angle=angle/57.29577951;

	dx=*x-xcen;
	dy=*y-ycen;

	*x=xcen+dx*cos(angle)+dy*sin(angle);
	*y=ycen-dx*sin(angle)+dy*cos(angle);
}

