; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
;+
; NAME:
;     widget_logger
;
; PURPOSE:
;     An object that implements a logging widget.
;
; TYPE:
;     OBJECT
;
; CATEGORY:
;     Widgets
;
; CALLING SEQUENCE:
;     myLogger = OBJ_NEW ('widget_logger')
;
; INPUTS:
;     NONE
;
; KEYWORDS:
;
;     XSIZE : xsize of the log text area (characters, DEFAULT = 60)
;     YSIZE : ysize of the log text area (characters, DEFAUTL = 10)
;     MAP   : 0 to prohibit mapping of the widget when the logger object is
;             instantiated, or 1 to map the widget (DEFAULT = 1)
;     TITLE : Logger widget window title (DEFAULT = 'Logger')
;             
; COMMON BLOCKS:
;     NONE
;
; SIDE EFFECTS:
;     None known.
;
; RESTRICTIONS:
;     None known.
;
; DEPENDENCIES:
;     NONE
;
; METHODS:
;     
;    append (PROCEDURE) - Append text to the logger widget
;
;         Inputs: text: STRING or STRARR of text to append to the logger
;        Outputs: NONE 
;       Keywords: NONE 
;
;    clear (PROCEDURE) - Clear the text area of the logger widget
;
;         Inputs: NONE
;        Outputs: NONE 
;       Keywords: NONE 
;
;    getText (FUNCTION) - Return the current text displayed in the logger
;
;         Inputs: NONE
;        Outputs: STRARR of text currently displayed
;       Keywords: NONE 
;
;    hide (PROCEDURE) - Hide (MAP = 0) the widget
;
;         Inputs: NONE
;        Outputs: NONE 
;       Keywords: NONE 
;
;    setClearButton (PROCEDURE) - Change the label of the "Clear" button
;
;         Inputs: STRING specifying new label for the button
;        Outputs: NONE
;       Keywords: NONE 
;
;    setHideButton (PROCEDURE) - Change the label of the "Hide" button
;
;         Inputs: STRING specifying new label for the button
;        Outputs: NONE
;       Keywords: NONE 
;
;    setLineNumber (PROCEDURE) - Set the zero-based line number of the 
;        line to be positioned on the topmost visible line in the text 
;        widget's viewport. No horizontal scrolling is performed. Note 
;        that this is a line number, not a character offset.
;
;         Inputs: INTEGER specifying the topmost visible line number
;        Outputs: NONE
;       Keywords: NONE 
;
;    setSize (PROCEDURE) - Set the XSIZE and/or YSIZE of the log text area
;
;         Inputs: NONE
;        Outputs: NONE 
;       Keywords: XSIZE : xsize of the log text area (characters)
;                 YSIZE : ysize of the log text area (characters)
;
;    show (PROCEDURE) - Show (MAP = 1) the widget
;
;         Inputs: NONE
;        Outputs: NONE 
;       Keywords: NONE 
;
;
;    getWidgetIDs (FUNCTION) - Return a stucture containing the widget ids 
;        of the logger components.  The following structure tags are returned:
;
;             top   : Top level base ID
;             text  : Text area widget ID
;             clear : Clear button widget ID
;             hide  : Hide button widget ID
;
;        DO NOT CHANGE THE UVALUES OF ANY OF THE LOGGER WIDGET COMPONENTS, AS
;        THEY ARE USED INTERNALLY BY THE OBJECT.  This method is not
;        recommended for use; manipulating the logger widget from outside
;        the object can compromise the internal object state.
;
;         Inputs: NONE
;        Outputs: STRUCTURE containing the logger widget component IDs 
;       Keywords: NONE 
;
; EXAMPLE:
;
;     ; Create a logger widget
;     ;
;     logger = OBJ_NEW ('widget_logger', MAP = 0, TITLE = 'myLogger')
;
;     ; Change the label of the 'Clear' button to 'Clear Log Text'
;     ;
;     logger->setClearButton, 'Clear Log Text'
;
;     ; Show the widget
;     ;
;     logger->show
;
;     ; Add some text
;     ;
;     logger->append, ['A line of text.', 'Another line of text.']
;
;     ; Clear the logger programatically
;     ;
;     logger->clear
;
;     ; Add a lot of text, then make the 300th line visible
;     ;
;     logger->append, 'Line number ' + STRTRIM (SINDGEN (500) + 1, 2)
;     logger->setLineNumber, 300
;
;     ; All done
;     ;
;     OBJ_DESTROY, logger
;
; MODIFICATION HISTORY:
;     v1.0: Written, 1998 June, Robert.Mallozzi@msfc.nasa.gov
;
;-
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 




; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Cleanup
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger::cleanup
    
    IF (WIDGET_INFO (self.topID, /VALID)) THEN $
       WIDGET_CONTROL, self.topID, /DESTROY

END



; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Initalize the logger
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
FUNCTION widget_logger::init, $
    XSIZE = xsize, YSIZE = ysize, MAP = map, TITLE = title


    ; By default, the logger is mapped on creation
    ;
    doMap = 1
    IF (N_ELEMENTS (map) NE 0) THEN $
       doMap = map
            
    self->build, XSIZE = xsize, YSIZE = ysize, TITLE = title

    IF (doMap) THEN $
       self->show

    
    RETURN, 1    
    
END ; init



; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Build the logger, and handle events.  Unfortunately, XMANAGER currently
; (v5.1) does not handle object methods correctly.  The following procedure
; forwards events to the logger object.
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger_EVENT, event

    WIDGET_CONTROL, event.top, GET_UVALUE = state
    state.self->display_event, event

END
; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
PRO widget_logger::display_event, event


    WIDGET_CONTROL, event.top, GET_UVALUE = state, /NO_COPY


    ; Widget button events only for this object
    ;
    WIDGET_CONTROL, event.id, GET_UVALUE = uvalue

    CASE (uvalue) OF

        'Clear': self->clear

        'Hide': self->hide
	
	ELSE: MESSAGE, /CONTINUE, 'Internal error, event not handled: ' + value
    
    ENDCASE

    WIDGET_CONTROL, event.top, SET_UVALUE = state, /NO_COPY


END ; display_event
; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
PRO widget_logger::build, XSIZE = xsize, YSIZE = ysize, TITLE = title


    IF (N_ELEMENTS (title) EQ 0) THEN $
       title = 'Logger'
       
    IF (N_ELEMENTS (xsize) EQ 0) THEN $
       xsize = 60

    IF (N_ELEMENTS (ysize) EQ 0) THEN $
       ysize = 10


    self.topID = WIDGET_BASE (TITLE = title, $
        /COLUMN, /BASE_ALIGN_CENTER, MAP = 0)

    self.textID = WIDGET_TEXT (self.topID, /SCROLL, $
        XSIZE = xsize, YSIZE = ysize)
    
    base = WIDGET_BASE (self.topID, ROW = 1, /GRID_LAYOUT)
    
    self.clearID = WIDGET_BUTTON (base, $
        VALUE = 'Clear', UVALUE = 'Clear', /DYNAMIC)
    self.hideID  = WIDGET_BUTTON (base, $
        VALUE = 'Hide', UVALUE = 'Hide', /DYNAMIC)
     
    state = { self: self } 


    ; Set data and realize
    ;
    WIDGET_CONTROL, self.topID, SET_UVALUE = state, /NO_COPY
    WIDGET_CONTROL, self.topID, /REALIZE


    XMANAGER, OBJ_CLASS (self), self.topID, /NO_BLOCK


END ; build


; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Show the logger
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger::show

    WIDGET_CONTROL, self.topID, MAP = 1 
           
END ; show


; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Hide the logger
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger::hide

    WIDGET_CONTROL, self.topID, MAP = 0 
    
END ; hide


; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Set the size of the logger
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger::setSize, XSIZE = xsize, YSIZE = ysize

    WIDGET_CONTROL, self.textID, XSIZE = xsize, YSIZE = ysize

END ; setSize



; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Set the topmost visible line number of the logger
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger::setLineNumber, lineNo

    IF (N_PARAMS () NE 1) THEN BEGIN
       MESSAGE, /CONTINUE, 'Usage: setLineNumber, number'
       RETURN
    ENDIF
       
    WIDGET_CONTROL, self.textID, SET_TEXT_TOP_LINE = FIX (lineNo)

END ; setLineNumber



; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Set the labels for the logger buttons
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger::setClearButton, newLabel

    IF (N_PARAMS () NE 1) THEN BEGIN
       MESSAGE, /CONTINUE, "Usage: setClearButton, 'label'"
       RETURN
    ENDIF
       
    WIDGET_CONTROL, self.clearID, SET_VALUE = STRING (newLabel)

END ; setClearButton

PRO widget_logger::setHideButton, newLabel

    IF (N_PARAMS () NE 1) THEN BEGIN
       MESSAGE, /CONTINUE, "Usage: setHideButton, 'label'"
       RETURN
    ENDIF
       
    WIDGET_CONTROL, self.hideID, SET_VALUE = STRING (newLabel)

END ; setHideButton



; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Clear the logger
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger::clear

    WIDGET_CONTROL, self.textID, SET_VALUE = ''

END ; clear



; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Append to the logger
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger::append, text

    WIDGET_CONTROL, self.textID, UPDATE = 0
    WIDGET_CONTROL, self.textID, SET_VALUE = text, /APPEND
    WIDGET_CONTROL, self.textID, UPDATE = 1
    
END ; append



; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Retrieve the current logger text
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
FUNCTION widget_logger::getText

    WIDGET_CONTROL, self.textID, GET_VALUE = currentText
    
    RETURN, currentText
    
END ; getText



; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Retrieve the logger widget IDs
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
FUNCTION widget_logger::getWidgetIDS

    ids = { $
        top:   self.topID, $
	text:  self.textID, $
        clear: self.clearID, $
        hide:  self.hideID $
    }    
    
    RETURN, ids
    
END ; getWidgetIDs



; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; Implementation of a widget logging object
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
PRO widget_logger__define

    obj = { widget_logger, $
    
        ; Widget IDs
	;
	topID:   0L, $
	textID:  0L, $
	clearID: 0L, $
	hideID:  0L  $

    }


END ; define
