; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
;+
; NAME:
;     DIALOG_SLIDER
;
; PURPOSE:
;     A modal (blocking) dialog widget to select a value from a slider.
;     The dialog must be dismissed, by activating the slider and pressing the
;     Return key, or by clicking on the 'Ok' or 'Cancel' button before
;     execution of the calling program can continue.
;
; TYPE:
;     FUNCTION
;
; CATEGORY:
;     WIDGETS
;
; CALLING SEQUENCE:
;     result = DIALOG_SLIDER (range)
;
; INPUTS:
;     range: Two element array giving the range for slider.
;
; KEYWORD PARAMETERS:
;     PROMPT: Text prompt for the slider [default = 'Select Value']
;
;     FLOAT: Set this keyword to use a floating point slider (default is
;            an integer slider.  Note that the floating point slider, which
;            uses CW_FSLIDER, does not update the slider label when the slider
;            is moved.  It does, however, allow entering of the slider value
;            through keyboard input. 
;
;     VERTICAL: Use a vertical slider [default = horizontal]
;
;     VALUE: Initial starting value [default = midpoint]
;
;     DIALOG_PARENT: Set this keyword to the widget ID of a widget over
;            which the message dialog should be positioned. When displayed,
;            the DIALOG_INPUT dialog will be positioned over the specified
;            widget. Dialogs are often related to a non-dialog widget tree.
;            The ID of the widget in that tree to which the dialog is most
;            closely related should be specified.
;
; OUTPUTS:
;     result: Selected value of the slider
;
; COMMON BLOCKS:
;     NONE
;
; SIDE EFFECTS:
;     Creates a modal widget
;
; RESTRICTIONS:
;     NONE
;
; DEPENDENCIES:
;     NONE
;
; MODIFICATION HISTORY:
;     v1.02: RSM, May 1998 changed TEXT keyword to PROMPT. 
;            NOT BACKWARD COMPATIBLE!
;     v1.01: RSM, Mar 1998, fixed error when used with a modal toplevel base.
;     v1.0: written, Robert.Mallozzi@msfc.nasa.gov, November 1997.
;
;-
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 

FUNCTION DIALOG_SLIDER, XR, $
    PROMPT = prompt, $
    FLOAT = float, $
    VERTICAL = vertical, $
    VALUE = VALUE, $
    DIALOG_PARENT = dialog_parent


N = SIZE (XR)
IF ((N[0] NE 1) OR (N[1] NE 2)) THEN BEGIN
   PRINT, 'DIALOG_SLIDER: Input range must be 2-element array.'
   RETURN, 0
ENDIF

HAVE_PROMPT = N_ELEMENTS (prompt) NE 0
HAVE_PARENT = N_ELEMENTS (dialog_parent) NE 0
HAVE_VALUE  = N_ELEMENTS (value) NE 0

; Top level base
;
IF (HAVE_PARENT) THEN BEGIN

   ; Check for a valid widget id
   ;
   HAVE_PARENT = WIDGET_INFO (dialog_parent, /VALID_ID)

ENDIF   

IF (HAVE_PARENT) THEN BEGIN
   s_base = WIDGET_BASE (TITLE = ' ', /COLUMN, $
       /FLOATING, /MODAL, GROUP_LEADER = dialog_parent)
ENDIF ELSE BEGIN
   s_base = WIDGET_BASE (TITLE = ' ', /COLUMN, MAP = 0)
ENDELSE

IF (HAVE_PROMPT) THEN BEGIN
   STEXT = WIDGET_LABEL (S_BASE, VALUE = prompt)
ENDIF ELSE BEGIN
   STEXT = WIDGET_LABEL (S_BASE, VALUE = 'Select Value')
ENDELSE

MIDDLE  = ((XR[1] - XR[0]) / 2.0) + XR[0]
IF (HAVE_VALUE) THEN $
   MIDDLE = VALUE

IF (KEYWORD_SET (float)) THEN BEGIN

   IF (KEYWORD_SET (vertical)) THEN BEGIN
      WSLIDER = CW_FSLIDER (S_BASE, TITLE = ' ', MIN = XR[0], MAX = XR[1], $
          UVALUE = 'WSLIDER', VALUE = MIDDLE, /EDIT, /VERTICAL)
   ENDIF ELSE BEGIN
      WSLIDER = CW_FSLIDER (S_BASE, TITLE = ' ', MIN = XR[0], MAX = XR[1], $
          UVALUE = 'WSLIDER', VALUE = MIDDLE, /EDIT)   
   ENDELSE

ENDIF ELSE BEGIN

   IF (KEYWORD_SET (vertical)) THEN BEGIN
      WSLIDER = WIDGET_SLIDER (S_BASE, TITLE = ' ', MIN = XR[0], MAX = XR[1], $
          UVALUE = 'WSLIDER', VALUE = MIDDLE, /VERTICAL)
   ENDIF ELSE BEGIN
      WSLIDER = WIDGET_SLIDER (S_BASE, TITLE = ' ', MIN = XR[0], MAX = XR[1], $
          UVALUE = 'WSLIDER', VALUE = MIDDLE)
   ENDELSE

ENDELSE

WIDGET_CONTROL, S_BASE, /REALIZE


; Place the dialog: window manager dependent
;
IF (NOT HAVE_PARENT) THEN BEGIN

   CURRENT_SCREEN = GET_SCREEN_SIZE()
   WIDGET_CONTROL, s_base, TLB_GET_SIZE = DIALOG_SIZE

   DIALOG_PT = [(CURRENT_SCREEN[0] / 2.0) - (DIALOG_SIZE[0] / 2.0), $ 
                (CURRENT_SCREEN[1] / 2.0) - (DIALOG_SIZE[1] / 2.0)] 

   WIDGET_CONTROL, s_base, $
                   TLB_SET_XOFFSET = DIALOG_PT[0], $
                   TLB_SET_YOFFSET = DIALOG_PT[1]
   WIDGET_CONTROL, s_base, MAP = 1

ENDIF

event = WIDGET_EVENT (s_base)

TYPE = TAG_NAMES (event, /STRUCTURE)
WIDGET_CONTROL, event.id, GET_VALUE = value, GET_UVALUE = uvalue

CASE (uvalue) OF 
     'WSLIDER': BEGIN
         SVALUE = value[0]
         WIDGET_CONTROL, event.top, /DESTROY
         END
ENDCASE



RETURN, SVALUE
END
