; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
;+
; NAME:
;     DIALOG_DRAW2
;
; PURPOSE:
;     A modal (blocking) dialog widget to display a plot.  
;     The dialog must be dismissed with the 'Dismiss' button before
;     execution of the calling program can continue.
;
; TYPE:
;     FUNCTION
;
; CATEGORY:
;     WIDGETS
;
; CALLING SEQUENCE:
;     result = DIALOG_PLOT (XDATA = xData, YDATA = yData)
;
; INPUTS:
;     XDATA and YDATA (via keyword, see below)
;
; KEYWORD PARAMETERS:
;
;     XDATA:         Array of independent data.
;     YDATA:         Array of dependent data.  
;
;     EXECUTE:       STRARR containing one or more IDL statements to execute. 
;                    The first command in the string must be a PLOT command.  
;                    The parameters xData, yData are available for use in the 
;                    EXECUTE string. See below for an example.
;  
;     PS_OPTION:     Add optional button to create a Postscript file
;     GIF_OPTION:    Add optional button to create a gif image
;
;     WINTITLE:      Window title [default = 'dialog_plot']
;
;     XSIZE:         The width of the plot region  (pixels) [default = 500]
;     YSIZE:         The height of the plot region (pixels) [default = 400]
;
;     DIALOG_PARENT: Set this keyword to the widget ID of a widget over
;                    which the message dialog should be positioned. When 
;                    displayed, the DIALOG_PLOT dialog will be positioned over 
;                    the specified widget. Dialogs are often related to a 
;                    non-dialog widget tree.  The ID of the widget in that 
;                    tree to which the dialog is most closely related should 
;                    be specified.
;
;     All PLOT keywords also accepted.
;
; OUTPUTS:
;     result: 1, or 0 for failure (missing parameters, or EXECUTE failed)
;
; COMMON BLOCKS:
;     NONE
;
; SIDE EFFECTS:
;     Creates a modal widget
;
; RESTRICTIONS:
;     NONE
;
; DEPENDENCIES:
;     NONE
;
; EXAMPLE:
;
;     xData = [0, 100]
;     yData = [-100, 100]
;
;     ; Create a simple plot, with an option to print to Postscript file.
;     ;   
;     result = DIALOG_PLOT (XDATA = xData, YDATA = yData, $
;         winTitle = 'Demo', /PS_OPTION)
;
;     ; Create a custom plot, with an option to print to gif file.  Note 
;     ; that EXECUTE can utilize the XDATA, YDATA keywords.
;     ;   
;     result = DIALOG_PLOT (XDATA = xData, YDATA = yData, EXECUTE = [ $
;         'PLOT, /NODATA, XDATA, YDATA', $
;         'OPLOT, [50, 50], !Y.CRANGE', $
;         'OPLOT, [70, 70], !Y.CRANGE, LINESTYLE = 1' ], $
;         XTITLE = 'xTitle', YTITLE = 'yTitle', winTitle = 'Demo', /GIF_OPTION)
;
; MODIFICATION HISTORY:
;     v0.1: written, Robert.Mallozzi@msfc.nasa.gov, March 1998.
;
;-
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 

FUNCTION DIALOG_DRAW2, Array, min, max,$
    EXECUTE = execute_s, $
    WINTITLE = winTitle, $
    XSIZE = xSize, YSIZE = ySize, $
    PS_OPTION = ps_option, ASCII_DUMP_OPTION = ascii_dump_option, $
    DIALOG_PARENT = dialog_parent, $
    _EXTRA = EXTRA, $
    XLINE=xline, YLINE=yline

haveArray= (N_ELEMENTS(Array) NE 0)

doExecute = (N_ELEMENTS(execute_s) NE 0)

IF (NOT haveArray) THEN BEGIN
   MESSAGE, /INFO, 'Usage: result = DIALOG_DRAW (ARRAY = array)'
   RETURN, 0
ENDIF

pSave = !P
xSave = !X
ySave = !Y

IF (N_ELEMENTS(winTitle) EQ 0) THEN $
   winTitle = 'dialog_plot'

HAVE_PARENT = N_ELEMENTS(dialog_parent) NE 0

usePsOption  = N_ELEMENTS(ps_option) NE 0
useAsciiDumpOption = N_ELEMENTS(ascii_dump_option) NE 0

IF (N_ELEMENTS(XSIZE) EQ 0) THEN BEGIN
   xSize = 500
ENDIF ELSE BEGIN
   xSize = XSIZE
ENDELSE

IF (N_ELEMENTS(YSIZE) EQ 0) THEN BEGIN
   ySize = 400
ENDIF ELSE BEGIN
   ySize = YSIZE
ENDELSE

; Top level base
;
IF (HAVE_PARENT) THEN BEGIN

   ; Check for a valid widget id
   ;
   HAVE_PARENT = WIDGET_INFO (LONG(dialog_parent), /VALID_ID)

ENDIF   

IF (HAVE_PARENT) THEN BEGIN
   base = WIDGET_BASE (TITLE = winTitle, /COLUMN, $
       /FLOATING, /MODAL, GROUP_LEADER = dialog_parent)
ENDIF ELSE BEGIN
   base = WIDGET_BASE (TITLE = winTitle, /COLUMN, MAP = 0)
ENDELSE

; Plotting canvas
;
winSave = !D.WINDOW
w = WIDGET_DRAW (base, XSIZE = xsize, YSIZE = ysize, /FRAME)
WIDGET_CONTROL, w, GET_VALUE = win
WSET, win

; Option buttons
;
IF (usePsOption) THEN $
   w = WIDGET_BUTTON (base, VALUE = 'Save to Postscript File')

IF (useAsciiDumpOption) THEN $
   w = WIDGET_BUTTON (base, VALUE = 'Save to ASCII File')

w = WIDGET_BUTTON (base, VALUE = 'Dismiss')


; Map to screen
;
WIDGET_CONTROL, base, /REALIZE


; Place the dialog: window manager dependent
;
IF (NOT HAVE_PARENT) THEN BEGIN

   CURRENT_SCREEN = GET_SCREEN_SIZE()
   WIDGET_CONTROL, base, TLB_GET_SIZE = DIALOG_SIZE

   DIALOG_PT = [(CURRENT_SCREEN[0] / 2.0) - (DIALOG_SIZE[0] / 2.0), $ 
                (CURRENT_SCREEN[1] / 2.0) - (DIALOG_SIZE[1] / 2.0)] 

   WIDGET_CONTROL, base, $
                   TLB_SET_XOFFSET = DIALOG_PT[0], $
                   TLB_SET_YOFFSET = DIALOG_PT[1]
   WIDGET_CONTROL, base, MAP = 1

ENDIF

; Plot data
;
status = 1
IF (doExecute) THEN BEGIN

   ; Build the scalar command required by IDL's EXECUTE function
   ;
   command = ''
   
   ; Pass along any additional keywords to the PLOT call
   ; TODO: verify that first command calls PLOT
   ;
   command = command + execute_s[0] + ', _EXTRA = EXTRA' 

   FOR i = 1, N_ELEMENTS(execute_s) - 1 DO $
       command = command + ' & ' + execute_s[i]

   status = EXECUTE (command)
   IF (NOT status) THEN BEGIN
      MESSAGE, /INFO, 'EXECUTE failed: ' + command
      GOTO, ERROR
   ENDIF
   
ENDIF ELSE BEGIN
   geom=widget_info(w,/geometry)
   print,geom
   tv, bytscl(Array,min=min,max=max)
;  for i=0,nstar-1 do plots, [xsize/nstar*(i+0.5),xsize/nstar*(i+0.5)],[0,ysize], COLOR=255*255*255, LINESTYLE=1, /device
;  plots, [0,xsize], [ysize*0.25,ysize*0.25], COLOR=255*255*255, LINESTYLE=1, /device
;  plots, [0,xsize], [ysize*0.75,ysize*0.75], COLOR=255*255*255, LINESTYLE=1, /device
   for i=0,n_elements(xline)-1 do plots, [xline[i],xline[i]], [0,ysize], COLOR=255*255*255, LINESTYLE=1, /device
   for i=0,n_elements(yline)-1 do plots, [0,xsize], [yline[i],yline[i]], COLOR=255*255*255, LINESTYLE=1, /device

ENDELSE

; Get the event, without using XMANAGER
;
NEXT:
event = WIDGET_EVENT (base)
WIDGET_CONTROL, event.id, GET_VALUE = value

IF (value EQ 'Save to Postscript File') THEN BEGIN

   fileName = DIALOG_PICKFILE (/WRITE, FILTER = '*.ps', $
       DIALOG_PARENT = event.top)
   IF (fileName NE '') THEN BEGIN

      d = FINDFILE (fileName, COUNT = fileExists)
      IF (fileExists) THEN BEGIN
         IF (DIALOG_MESSAGE (/QUESTION, ['File ' + fileName + ' exists.', $
	     'Overwrite?'], DIALOG_PARENT = event.top) EQ 'No') THEN $
            GOTO, NEXT
      ENDIF

      deviceSave = !D.NAME
      SET_PLOT, 'PS'
      DEVICE, FILE = fileName

;	status = EXECUTE (command)

      PLOT, xData, yData, _EXTRA = EXTRA
      device,/close
      SET_PLOT, deviceSave
 

   ENDIF

   GOTO, NEXT
   
ENDIF

IF (value EQ 'Save to ASCII File') THEN BEGIN

   fileName = DIALOG_PICKFILE (/WRITE, FILTER = '*.xy', $
       DIALOG_PARENT = event.top)
   IF (fileName NE '') THEN BEGIN

      d = FINDFILE (fileName, COUNT = fileExists)
      IF (fileExists) THEN BEGIN
         IF (DIALOG_MESSAGE (/QUESTION, ['File ' + fileName + ' exists.', $
	     'Overwrite?'], DIALOG_PARENT = event.top) EQ 'No') THEN $
            GOTO, NEXT
      ENDIF
	openw,1,fileName
;	FOR count=0,size(XDATA,/N_ELEMENTS)-1 DO BEGIN
	ndata=size(XDATA)
	nndata=ndata(3)
	FOR count=0,nndata-1 DO BEGIN
	printf,1,XDATA[count],YDATA[count]
	ENDFOR
	close,1
      
   ENDIF

   GOTO, NEXT
   
ENDIF


ERROR:

!P = pSave
!X = xSave
!Y = ySave

WSET, winSave


WIDGET_CONTROL, base, /DESTROY
RETURN, status



END
